/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "gui_minimal.h"
#include "ui_gui_minimal.h"

#include "miscellaneous.h"
#include "serial_v2.h"

#include <QDebug>
#include <QMessageBox>

GUI_Minimal::GUI_Minimal(QWidget *parent) : QMainWindow(parent), ui(new Ui::GUI_Minimal)
{
	ui->setupUi(this);

	firmware_version_requirement[0] = 1;
	firmware_version_requirement[1] = 6;
	firmware_version_requirement[2] = 4;
	firmware_version_requirement[3] = 0;
	toolkit_GUI_version = "0.4.1";
	qDebug() << "Toolkit GUI " << toolkit_GUI_version;
//	ui->GUI_version_label_2->setText(toolkit_GUI_version);

	/* set Home menu default */
	ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->page_home));

	/* get the names of the serialports, as well as all the other startup values from config.txt */
	config = new ConfigHandler(QCoreApplication::applicationDirPath() + "/config.txt");

	//Setup Data Logger
//	QStringList loggercolumns = {"RF_enable", "Frequency", "Power_Setpoint_watt", "Power_FWD_watt", "Power_RFL_watt", "S11_Reflection", "PWM_Frequency", "PWM_Duty_Cycle", "PA_Temperature", "SGx_Error"};
//	datalogger = new LoggingClass(config->get_logging_directory(), "log", loggercolumns);


	/*******************************************************************
	 * Setup channels
	 *******************************************************************/
	RF_system_constructor = new RF_System_Minimal();

	/* Activate Channels */
	for (int i = 0; i < RF_System::Channels->count(); i++)
	{
		connect(this, &GUI_Minimal::signal_channelInit, RF_System::Channels->at(i), &RF_Channel::Initialize);
		connect(this, &GUI_Minimal::signal_timerStart, RF_System::Channels->at(i), &RF_Channel::Timer_Readings_Start);
		connect(this, &GUI_Minimal::signal_timerStop, RF_System::Channels->at(i), &RF_Channel::Timer_Readings_Stop);
		connect(this, &GUI_Minimal::signal_execute_reset_SGx, RF_System::Channels->at(i), &RF_Channel::Execute_Reset_SGx);
		connect(this, &GUI_Minimal::signal_execute_error_clear, RF_System::Channels->at(i), &RF_Channel::Execute_Error_Clear);
		connect(this, &GUI_Minimal::signal_set_error_clearing_enable, RF_System::Channels->at(i), &RF_Channel::Set_Error_Clearing_Enable);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_firmware_version_get, this, &GUI_Minimal::handler_firmware_version_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_error_get, this, &GUI_Minimal::handler_error_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_reset_detected, this, &GUI_Minimal::handler_reset_detected);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_temperature_PA_get, this, &GUI_Minimal::handler_temperature_get);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_RF_enable_get, this, &GUI_Minimal::handler_RF_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PA_power_readings, this, &GUI_Minimal::handler_PA_power_readings);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_power_get, this, &GUI_Minimal::handler_power_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_frequency_get, this, &GUI_Minimal::handler_frequency_get);

		connect(this, &GUI_Minimal::signal_request_PA_power_update, RF_System::Channels->at(i), &RF_Channel::Update_PA_Power);

		connect(this, &GUI_Minimal::signal_RF_enable, RF_System::Channels->at(i), &RF_Channel::Set_RF_enable);
		connect(this, &GUI_Minimal::signal_set_power_control_mode, RF_System::Channels->at(i), &RF_Channel::Set_Power_Control_Mode);

		connect(this, &GUI_Minimal::signal_setPowerWatt, RF_System::Channels->at(i), &RF_Channel::Set_Power_Watt);
		connect(this, &GUI_Minimal::signal_setPowerdBm, RF_System::Channels->at(i), &RF_Channel::Set_Power_dBm);
		connect(this, &GUI_Minimal::signal_setFrequency, RF_System::Channels->at(i), &RF_Channel::Set_Frequency);

//		connect(this, &GUI_Minimal::signal_execute_SWP, RF_System::Channels->at(i), &RF_Channel::Execute_SWP_dBm);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_channel_working, this, &GUI_Minimal::handler_channel_working);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_channel_init_failed, this, &GUI_Minimal::handler_channel_init_failed);

//		connect(datalogger, &LoggingClass::signal_datalogging_enable, this, &GUI_Minimal::handler_datalogging_enable_get);
//		connect(datalogger, &LoggingClass::signal_datalogging_storage_sufficient, this, &GUI_Minimal::handler_datalogging_storage_sufficient);
//		connect(datalogger, &LoggingClass::signal_datalogging_log_deleted, this, &GUI_Minimal::handler_datalogging_log_deleted);

		emit signal_channelInit(i+1);
		emit signal_timerStart(i+1, RF_system_constructor->config->get_polling_rate_data());
	}

	/* set up button groups, make it so only one button can be checked at a time */
	buttonGroup_RF_Enable.addButton(ui->pushButton_RF_ON_1);
	buttonGroup_RF_Enable.addButton(ui->pushButton_RF_OFF_1);
	buttonGroup_RF_Enable.setExclusive(true);

//	logging_buttonGroup.addButton(ui->pushButton_data_logging_OFF_1);
//	logging_buttonGroup.addButton(ui->pushButton_data_logging_ON_1);
//	logging_buttonGroup.setExclusive(true);


	//
	// TODO:
	// Get TCP RCM working and Remove these plz.
	//
	/* Initiate the UART connection & prepare for TCP connection. */
	serial_Setup(RCM_port, config->get_RCM_port_USB());	//Port for external USB communication.
	serial_Setup(SG_port, config->get_output_port());	//Port for SGx connection.
	tcpServer = new QTcpServer(this);
	tcpSocket = new QTcpSocket(this);

	/* Create Numpad object + connect input confirmation with the respective handler */
	numpad = new Numpad(5,4, ui->numpadLabel_display, ui->numpadLabel_unit, ui->numpadButton_period, ui->numpadButton_sign);
	connect(numpad, &Numpad::value_confirmed_signal, this, &GUI_Minimal::numpad_value_confirmed_handler);

	/* optionally enable the temperature measurements */
	ui->label_temperature_1->setVisible(config->get_read_temperature() >= 1);
	ui->label_temperature_2->setVisible(config->get_read_temperature() >= 1);
	ui->label_title_temperature_1->setVisible(config->get_read_temperature() >= 1);

	RCM_supported = config->get_support_RCM_mode_USB_live();
	ui->pushButton_remote_command_USB_1->setEnabled(config->get_support_RCM_mode_USB_live());

	// Set the start condition for RCM.
	handler_RCM_mode();

	// Set the start condition for Logging
//	if (config->get_support_logging())
//	{
//		handler_datalogging(config->get_logging_enabled());
//	}
}

GUI_Minimal::~GUI_Minimal()
{
	delete ui;
}


void GUI_Minimal::firmware_version_check(QString firmware_version)
{
	QStringList ver = firmware_version.split(".");
	int firmware_version_actual[4];
	firmware_version_actual[0] = ver.at(0).toInt();
	firmware_version_actual[1] = ver.at(1).toInt();
	firmware_version_actual[2] = ver.at(2).toInt();
	if (ver.count() > 3)
	{
		firmware_version_actual[3] = ver.at(3).toInt();
	}
	else
	{
		firmware_version_actual[3] = 0;
	}

	if(isVersionCompatible(firmware_version_actual,firmware_version_requirement) == false)
	{
		QString comp_warning =	"The current firmware version is unsupported by this application and may result in unpredictable behaviour. "
								"Please upgrade the firmware of your SGx board to at least the recommended version.\n"
								"\nRecommended firmware: v";
		for (int i = 0; i < sizeof(&firmware_version_requirement); i++)
		{
			comp_warning += QString::number(firmware_version_requirement[i]) + ".";
		}
		comp_warning.chop(1);	//chop the redundant period
		comp_warning += "\nCurrent firmware: " + firmware_version +
						"\n\nProceed at own risk.";

		QMessageBox message;
		message.warning(nullptr,	"Software imcompatibility", comp_warning);
//		show_notification("Firmware is outdated!\nProceed at own risk.");
	}
}





void GUI_Minimal::handler_firmware_version_get(int intrasys_num, int channel_num, QString version)
{
	if (version != "")
	{
		firmware_version_check(version);
//		ui->firmware_version_label_2->setText(version);
	}
}


//Miscellaneous
void GUI_Minimal::handler_channel_working(int channel_number, bool enabled)
{
	this->setEnabled(enabled);
}

void GUI_Minimal::handler_channel_init_failed(int intrasys_num)
{
	qDebug() << "GUI: Subsystem[" << intrasys_num << "] Initialization failed.";
	if (RF_system_constructor->config->get_init_failed_exit() == true)
	{
		qDebug() << "Exiting GUI.";
		this->close();
	}
}


void GUI_Minimal::handler_error_get(int intrasys_num, quint64 error, QStringList error_messages)
{
	if (error > 0)
	{
		//display in error label.
		ui->label_error_2->setText(QString("0x%1").arg(error,0,16));

		QString message_string = "";
		for (int i = 0; i < error_messages.count(); i++)
		{
			if (error_messages.at(i) != "")
			{
				message_string += error_messages.at(i);
			}
			if (i != error_messages.count() - 1)
			{
				message_string += "\n";
			}
		}

//		show_error(message_string);
	}
	else
	{
		ui->label_error_2->setText(error_messages.at(0));
//		close_error();
	}

	//datalogging
//	datalogger->setData(9, QString::number(error,16));
}

void GUI_Minimal::handler_reset_detected(int intrasys_num, int channel_number)
{
//	QMessageBox message;
//	message.critical(nullptr,"Reset Detected", "A reset has occurred.\nSystem has been reset to defaults");
}

void GUI_Minimal::handler_temperature_get(int intrasys_num, double temperature)
{
	ui->label_temperature_1->setText(QString::number(temperature,'f',1));
}

void GUI_Minimal::handler_RCM_mode(){
	if (RCM_supported == true)
	{
		switch(config->get_remote_command_mode())
		{
			case 0:
//				ui->pushButton_remote_command_OFF_1->click();
				break;
			case 1:
				ui->pushButton_remote_command_USB_1->click();
				break;
			case 2:
//				ui->pushButton_remote_command_TCP_1->click();
				break;
			default:
//				ui->pushButton_remote_command_OFF_1->click();
				break;
		}
	}
}

void GUI_Minimal::handler_RF_enable_get(int intrasys_num, bool RF_enabled)
{
	if (RF_enabled == true)
	{
		ui->pushButton_RF_ON_1->setChecked(true);

		//datalogging
//		datalogger->setData(0, "true");
	}
	else
	{
		ui->pushButton_RF_OFF_1->setChecked(true);

		//datalogging
//		datalogger->setData(0, "false");
	}
}

void GUI_Minimal::handler_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_reflection)
{
	double PA_power_minimum_dbm = 30;

	ui->label_power_forward_1->setText(QString::number(PA_power_fwd_watt,'f',1));
	ui->label_power_reflected_1->setText(QString::number(PA_power_rfl_watt,'f',1));

	if (PA_power_fwd_dbm >= PA_power_minimum_dbm)
	{
		ui->label_S11_1->setText(QString::number(PA_s11_reflection,'f',1));
	}
	else
	{
		ui->label_S11_1->setText("-");
	}

	//datalogging
//	datalogger->setData(3, QString::number(PA_power_fwd_watt,'f',1));
//	datalogger->setData(4, QString::number(PA_power_rfl_watt,'f',1));
//	datalogger->setData(5, QString::number(PA_s11_reflection,'f',1));
}

void GUI_Minimal::handler_power_get(int intrasys_num, double power_dbm, double power_watt)
{
	ui->pushButton_power_watt_1->setText(zeroChopper(QString::number(power_watt, 'f', 2)));

	//datalogging
//	datalogger->setData(2, QString::number(power_watt));
}

void GUI_Minimal::handler_frequency_get(int intrasys_num, double frequency_mhz)
{
	ui->pushButton_frequency_1->setText(zeroChopper(QString::number(frequency_mhz, 'f', 2)));

	//datalogging
//	datalogger->setData(1, QString::number(frequency_mhz));
}

void GUI_Minimal::on_pushButton_RF_OFF_1_clicked()
{
	emit signal_RF_enable(1, false);
}

void GUI_Minimal::on_pushButton_RF_ON_1_clicked()
{
	emit signal_RF_enable(1, true);
}

void GUI_Minimal::on_pushButton_frequency_1_clicked()
{
	numpad_value = RF_System::Channels->at(0)->Frequency();
	numpad_parameter_select(ui->pushButton_frequency_1, &numpad_value, "MHz", 3);
}

void GUI_Minimal::on_pushButton_power_watt_1_clicked()
{
	numpad_value = RF_System::Channels->at(0)->Power_watt();
	numpad_parameter_select(ui->pushButton_power_watt_1, &numpad_value, "watt", 1);
}

/**********************************************************************************************************************************************************************************
 * NUMPAD BUTTONS
 * *******************************************************************************************************************************************************************************/

void GUI_Minimal::on_numpadButton_0_clicked(){
	numpad->press_0();
}
void GUI_Minimal::on_numpadButton_1_clicked(){
	numpad->press_1();
}
void GUI_Minimal::on_numpadButton_2_clicked(){
	numpad->press_2();
}
void GUI_Minimal::on_numpadButton_3_clicked(){
	numpad->press_3();
}
void GUI_Minimal::on_numpadButton_4_clicked(){
	numpad->press_4();
}
void GUI_Minimal::on_numpadButton_5_clicked(){
	numpad->press_5();
}
void GUI_Minimal::on_numpadButton_6_clicked(){
	numpad->press_6();
}
void GUI_Minimal::on_numpadButton_7_clicked(){
	numpad->press_7();
}
void GUI_Minimal::on_numpadButton_8_clicked(){
	numpad->press_8();
}
void GUI_Minimal::on_numpadButton_9_clicked(){
	numpad->press_9();
}
void GUI_Minimal::on_numpadButton_period_clicked(){
	numpad->press_period();
}
void GUI_Minimal::on_numpadButton_backspace_clicked(){
	numpad->press_delete();
}
void GUI_Minimal::on_numpadButton_clear_all_clicked(){
	numpad->press_delete_all();
}
void GUI_Minimal::on_numpadButton_arrow_left_clicked(){
	numpad->press_indicator_left();
}
void GUI_Minimal::on_numpadButton_arrow_right_clicked(){
	numpad->press_indicator_right();
}
void GUI_Minimal::on_numpadButton_plus_clicked(){
	numpad->press_plus();
}
void GUI_Minimal::on_numpadButton_minus_clicked(){
	numpad->press_minus();
}
void GUI_Minimal::on_numpadButton_ok_clicked(){
	numpad->press_confirm_input();
	ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->page_home));
}

/**********************************************************************************************************************************************************************************
 * FUNCTIONS FOR NUMPAD
 * *******************************************************************************************************************************************************************************/
void GUI_Minimal::numpad_parameter_select(QPushButton *button, double *parameter, QString unit, int precision, double scale,  bool period_enable, bool sign_enable)
{
	numpad->setOutputButton(button);
	numpad->load_value(parameter, unit, period_enable, sign_enable, scale);
	ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->page_numpad));
	scale_multiplier = scale;
	this->precision = precision;
}

void GUI_Minimal::numpad_value_confirmed_handler(QPushButton *button, double *num)
{
	QString str = zeroChopper(QString::number(*num * scale_multiplier,'f', precision));
	button->setText(str);

	//Home Page parameters
	if (button == ui->pushButton_frequency_1){
		emit signal_setFrequency(1, *num);
	}
	else if (button == ui->pushButton_power_watt_1){
		emit signal_setPowerWatt(1, *num);
	}
}

// Live Remote Command.
/* Enable 'Live' RCM. (send commands, while the UI remains usable) */
void GUI_Minimal::on_pushButton_remote_command_USB_1_clicked()
{
	if (!RF_system_constructor->RCM_USB_port->rcm_port->isOpen())
	{
		if (!RF_system_constructor->RCM_USB_port->Enable())
		{
			ui->pushButton_remote_command_USB_1->setChecked(false);
		}
	}
	else
	{
		RF_system_constructor->RCM_USB_port->Disable();
		ui->pushButton_remote_command_USB_1->setChecked(false);

		if (config->get_reset_on_startup() == true)
		{
			emit signal_execute_reset_SGx(1);
		}
	}

	configure_auto_error_clearing();
}

/* Enable or disable automatic error clearing */
void GUI_Minimal::configure_auto_error_clearing()
{
	bool enable = !ui->pushButton_remote_command_USB_1->isChecked();
	emit signal_set_error_clearing_enable(1, enable);
}
